import os, csv, yaml, itertools as it, importlib.util, sys, pathlib

ROOT = pathlib.Path(".")
CFG = yaml.safe_load(open("configs/default.yaml","r",encoding="utf-8"))

def get(cfg, *cands, required=True, default=None):
    for path in cands:
        node = cfg
        ok = True
        for k in path.split("."):
            if isinstance(node, dict) and k in node: node = node[k]
            else: ok = False; break
        if ok: return node
    if required: raise KeyError(f"None of {cands} found in config")
    return default

Ls      = get(CFG, "adjoint_volume.volumes", "L_values", "L_list")
gauges  = get(CFG, "adjoint_volume.gauge_groups", "adjoint_volume.gauges")
bs      = get(CFG, "b_values", "adjoint_volume.b_values")
ks      = get(CFG, "k_values", "adjoint_volume.k_values")
n0s     = get(CFG, "n0_values", "adjoint_volume.n0_values")
fit_range = get(CFG, "adjoint_volume.fit_range", required=False, default=[1,3])
kpaths    = get(CFG, "adjoint_volume.kernel_paths")
flip_tpl  = get(CFG, "adjoint_volume.flip_counts_path_template", required=False)

boot_reps  = get(CFG, "adjoint_volume.bootstrap_reps",  required=False, default=400)
boot_block = get(CFG, "adjoint_volume.bootstrap_block", required=False, default=0)
boot_seed  = get(CFG, "adjoint_volume.bootstrap_seed",  required=False, default=1337)
hash_guard = get(CFG, "adjoint_volume.kernel_hash_guard", required=False, default=True)

# SAFE IMPORT (register module before exec)
mod_path = pathlib.Path("orig/simulation/compute_tension.py")
spec = importlib.util.spec_from_file_location("fphs_compute_tension", mod_path)
ct_mod = importlib.util.module_from_spec(spec)
sys.modules["fphs_compute_tension"] = ct_mod
spec.loader.exec_module(ct_mod)

compute_with_err = getattr(ct_mod, "compute_tension_with_err", None)
compute_scalar   = getattr(ct_mod, "compute_string_tension")

out_dir = ROOT / "data" / "results" / "vol4_wilson_loop_adjoint_volume_sweep"
out_dir.mkdir(parents=True, exist_ok=True)
out_csv = out_dir / "adjoint_volume_summary.csv"

rows = []
for L, gauge, b, k, n0 in it.product(Ls, gauges, bs, ks, n0s):
    kernel_path = kpaths[gauge].format(L=L)
    flip_path = flip_tpl.format(L=L) if flip_tpl else None

    kw = dict(b=float(b), k_exp=float(k), n0=float(n0),
              L=int(L), gauge=str(gauge),
              volumes=Ls, fit_range=fit_range,
              kernel_path=kernel_path, flip_counts_path=flip_path)

    if compute_with_err is not None:
        sigma, sigma_err, meta = compute_with_err(
            **kw,
            bootstrap_reps=int(boot_reps),
            bootstrap_block=int(boot_block),
            bootstrap_seed=int(boot_seed),
            kernel_hash_guard=bool(hash_guard),
        )
        st = sigma / (2 * (L**2))
        sterr = sigma_err / (2 * (L**2))
        rows.append({
            "L":L,"b":b,"gauge":gauge,"k":k,"n0":n0,
            "sigma":sigma,"sigma_err":sigma_err,
            "string_tension":st,"string_tension_err":sterr,
            "err_strategy":meta.get("strategy","kernel_block_bootstrap"),
            "err_reps":meta.get("reps",0),"err_block":meta.get("block",0),
            "flipcount_n":meta.get("n",0),
            "kernel_sha256":meta.get("kernel_sha256"),
            "kernel_path":meta.get("kernel_path"),
            "flip_counts_path":meta.get("flip_counts_path"),
        })
    else:
        sigma = compute_scalar(**kw)
        st = sigma / (2 * (L**2))
        rows.append({
            "L":L,"b":b,"gauge":gauge,"k":k,"n0":n0,
            "sigma":sigma,"sigma_err":0.0,
            "string_tension":st,"string_tension_err":0.0,
            "err_strategy":"none","err_reps":0,"err_block":0,
            "flipcount_n":0, "kernel_sha256":None,
            "kernel_path":kw["kernel_path"], "flip_counts_path":kw.get("flip_counts_path"),
        })

cols = ["L","b","gauge","k","n0","sigma","sigma_err","string_tension","string_tension_err",
        "err_strategy","err_reps","err_block","flipcount_n","kernel_sha256","kernel_path","flip_counts_path"]
with open(out_csv, "w", newline="", encoding="utf-8") as f:
    w = csv.DictWriter(f, fieldnames=cols)
    w.writeheader(); w.writerows(rows)

print(f"Wrote {len(rows)} rows → {out_csv}")
